<?php
// Include required files
require_once 'includes/cors.php';
require_once 'includes/response.php';
require_once 'config/database.php';

// Get request method and path
$method = $_SERVER['REQUEST_METHOD'];
$request_uri = $_SERVER['REQUEST_URI'];

// Parse the path - extract endpoint after /api/
$path = parse_url($request_uri, PHP_URL_PATH);
$path_parts = explode('/', trim($path, '/'));

// Find 'api' in path and get the endpoint after it
$endpoint = isset($_GET['endpoint']) ? $_GET['endpoint'] : '';

// Route requests
try {
    switch($endpoint) {
        case '':
        case 'index.php':
            handleRoot();
            break;
        case 'menu':
            handleMenu($method);
            break;
        case 'order':
            handleOrder($method);
            break;
        case 'claim':
            handleClaim($method);
            break;
        default:
            sendError("Endpoint not found: /$endpoint", 404);
    }
} catch(Exception $e) {
    sendError("Server error: " . $e->getMessage(), 500);
}

// ========================================
// ROOT ENDPOINT
// ========================================
function handleRoot() {
    sendResponse(true, [
        'app' => 'Digiteen API',
        'version' => '1.0.0',
        'endpoints' => [
            'GET /api/menu' => 'Get menu items',
            'POST /api/order' => 'Place new order',
            'POST /api/claim' => 'Claim order and print'
        ]
    ], "API is running", 200);
}

// ========================================
// MENU ENDPOINT
// ========================================
function handleMenu($method) {
    if ($method !== 'GET') {
        sendError("Method not allowed. Use GET", 405);
    }

    try {
        $database = new Database();
        $db = $database->getConnection();

        if (!$db) {
            sendError("Database connection failed", 500);
        }

        // Get optional category filter
        $category = isset($_GET['category']) ? $_GET['category'] : null;

        $query = "SELECT id, name, description, price, category, image_url, available 
                  FROM menu_items 
                  WHERE available = 1";
        
        if ($category) {
            $query .= " AND category = :category";
        }
        
        $query .= " ORDER BY category, name";

        $stmt = $db->prepare($query);
        
        if ($category) {
            $stmt->bindParam(":category", $category);
        }
        
        $stmt->execute();
        $items = $stmt->fetchAll(PDO::FETCH_ASSOC);

        sendResponse(true, $items, "Menu fetched successfully", 200);

    } catch(Exception $e) {
        sendError("Error fetching menu: " . $e->getMessage(), 500);
    }
}

// ========================================
// ORDER ENDPOINT
// ========================================
function handleOrder($method) {
    if ($method !== 'POST') {
        sendError("Method not allowed. Use POST", 405);
    }

    try {
        // Get posted data
        $json = file_get_contents("php://input");
        $data = json_decode($json);

        if (!$data) {
            sendError("Invalid JSON data", 400);
        }

        // Validate required fields
        if (empty($data->items) || !isset($data->total_amount)) {
            sendError("Missing required fields: items, total_amount", 400);
        }

        $database = new Database();
        $db = $database->getConnection();

        if (!$db) {
            sendError("Database connection failed", 500);
        }

        // Generate unique order ID
        $orderId = 'ORD' . date('Ymd') . rand(1000, 9999);
        
        // Check if order ID already exists
        $checkQuery = "SELECT order_id FROM orders WHERE order_id = :order_id";
        $checkStmt = $db->prepare($checkQuery);
        $checkStmt->bindParam(":order_id", $orderId);
        $checkStmt->execute();
        
        // Regenerate if duplicate
        while ($checkStmt->rowCount() > 0) {
            $orderId = 'ORD' . date('Ymd') . rand(1000, 9999);
            $checkStmt->bindParam(":order_id", $orderId);
            $checkStmt->execute();
        }

        // Get order time
        $orderTime = isset($data->order_time) ? $data->order_time : date('Y-m-d H:i:s');

        // Start transaction
        $db->beginTransaction();

        // Insert order
        $orderQuery = "INSERT INTO orders (order_id, total_amount, status, order_time) 
                       VALUES (:order_id, :total_amount, 'pending', :order_time)";
        $orderStmt = $db->prepare($orderQuery);
        $orderStmt->bindParam(":order_id", $orderId);
        $orderStmt->bindParam(":total_amount", $data->total_amount);
        $orderStmt->bindParam(":order_time", $orderTime);
        $orderStmt->execute();

        // Insert order items
        $itemQuery = "INSERT INTO order_items (order_id, item_id, name, price, quantity) 
                      VALUES (:order_id, :item_id, :name, :price, :quantity)";
        $itemStmt = $db->prepare($itemQuery);

        foreach ($data->items as $item) {
            $itemStmt->bindParam(":order_id", $orderId);
            $itemStmt->bindParam(":item_id", $item->item_id);
            $itemStmt->bindParam(":name", $item->name);
            $itemStmt->bindParam(":price", $item->price);
            $itemStmt->bindParam(":quantity", $item->quantity);
            $itemStmt->execute();
        }

        // Commit transaction
        $db->commit();

        // Return order details
        $response = [
            "order_id" => $orderId,
            "total_amount" => $data->total_amount,
            "order_time" => $orderTime,
            "status" => "pending"
        ];

        sendResponse(true, $response, "Order placed successfully", 201);

    } catch(Exception $e) {
        // Rollback on error
        if (isset($db)) {
            $db->rollBack();
        }
        sendError("Error placing order: " . $e->getMessage(), 500);
    }
}

// ========================================
// CLAIM ENDPOINT
// ========================================
function handleClaim($method) {
    if ($method !== 'POST') {
        sendError("Method not allowed. Use POST", 405);
    }

    try {
        // Get posted data
        $json = file_get_contents("php://input");
        $data = json_decode($json);

        if (!$data) {
            sendError("Invalid JSON data", 400);
        }

        // Validate order_id
        if (empty($data->order_id)) {
            sendError("Missing required field: order_id", 400);
        }

        $database = new Database();
        $db = $database->getConnection();

        if (!$db) {
            sendError("Database connection failed", 500);
        }

        // Check if order exists
        $checkQuery = "SELECT id, status FROM orders WHERE order_id = :order_id";
        $checkStmt = $db->prepare($checkQuery);
        $checkStmt->bindParam(":order_id", $data->order_id);
        $checkStmt->execute();

        if ($checkStmt->rowCount() == 0) {
            sendError("Order not found", 404);
        }

        $order = $checkStmt->fetch(PDO::FETCH_ASSOC);

        // Check if already claimed
        if ($order['status'] === 'claimed' || $order['status'] === 'completed') {
            sendError("Order already claimed", 400);
        }

        // Update order status
        $updateQuery = "UPDATE orders 
                        SET status = 'claimed', claimed_time = NOW() 
                        WHERE order_id = :order_id";
        $updateStmt = $db->prepare($updateQuery);
        $updateStmt->bindParam(":order_id", $data->order_id);
        $updateStmt->execute();

        // Get order details for printing
        $orderQuery = "SELECT o.order_id, o.total_amount, o.order_time, o.claimed_time,
                              oi.name, oi.price, oi.quantity
                       FROM orders o
                       LEFT JOIN order_items oi ON o.order_id = oi.order_id
                       WHERE o.order_id = :order_id";
        $orderStmt = $db->prepare($orderQuery);
        $orderStmt->bindParam(":order_id", $data->order_id);
        $orderStmt->execute();
        $orderDetails = $orderStmt->fetchAll(PDO::FETCH_ASSOC);

        // TODO: Trigger print to Raspberry Pi
        // Example: file_get_contents("http://raspberry-pi-ip/print.php?order_id=".$data->order_id);
        
        error_log("Printing order: " . $data->order_id);

        $response = [
            "order_id" => $data->order_id,
            "status" => "claimed",
            "print_status" => "sent_to_printer",
            "order_details" => $orderDetails
        ];

        sendResponse(true, $response, "Order claimed and sent to printer", 200);

    } catch(Exception $e) {
        sendError("Error claiming order: " . $e->getMessage(), 500);
    }
}
?>