<?php
/**
 * SECURE ORDERS MANAGEMENT
 * admin/orders.php - Fixed SQL injection vulnerabilities
 */
session_start();
require_once '../config/database.php';
require_once 'auth.php'; // Use the authentication middleware

$conn = getConnection();

// Handle status update with CSRF protection
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['update_status'])) {
    // Validate CSRF token
    if (!isset($_POST['csrf_token']) || !validateCSRFToken($_POST['csrf_token'])) {
        die('Invalid CSRF token');
    }
    
    // Sanitize and validate inputs
    $order_id = sanitizeInput($_POST['order_id']);
    $status = sanitizeInput($_POST['status']);
    
    // Validate status is allowed
    $allowed_statuses = ['pending', 'processing', 'shipped', 'delivered', 'cancelled'];
    if (!in_array($status, $allowed_statuses)) {
        $error = "Invalid status value";
    } else {
        // Use prepared statement
        $stmt = $conn->prepare("UPDATE orders SET status = ? WHERE order_id = ?");
        $stmt->bind_param("ss", $status, $order_id);
        
        if ($stmt->execute()) {
            $success = "Order status updated successfully!";
            
            // Optional: Send email notification
            // $emailService->sendStatusUpdate($order, $status);
        } else {
            $error = "Failed to update order status";
        }
        $stmt->close();
    }
}

// Get filter with whitelist validation
$filter = isset($_GET['status']) ? sanitizeInput($_GET['status']) : 'all';
$allowed_filters = ['all', 'pending', 'processing', 'shipped', 'delivered', 'cancelled'];

if (!in_array($filter, $allowed_filters)) {
    $filter = 'all';
}

// Build query with prepared statement
$sql = "SELECT * FROM orders";
$params = [];
$types = "";

if ($filter !== 'all') {
    $sql .= " WHERE status = ?";
    $params[] = $filter;
    $types = "s";
}

$sql .= " ORDER BY created_at DESC";

$stmt = $conn->prepare($sql);
if (!empty($params)) {
    $stmt->bind_param($types, ...$params);
}
$stmt->execute();
$orders = $stmt->get_result();

// Get statistics with prepared statements
$stats = [];
$stmt = $conn->query("SELECT 
    COUNT(*) as total_orders,
    SUM(CASE WHEN status = 'pending' THEN 1 ELSE 0 END) as pending_orders,
    SUM(CASE WHEN status = 'delivered' THEN 1 ELSE 0 END) as delivered_orders,
    SUM(total_amount) as total_revenue
FROM orders");
$stats = $stmt->fetch_assoc();
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Orders Management - Boutique Admin</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <!-- Include your existing styles -->
</head>
<body>
    <?php include 'navbar.php'; ?>

    <div class="container">
        <?php include 'sidebar.php'; ?>

        <main class="main-content">
            <!-- Success/Error Messages -->
            <?php if (isset($success)): ?>
                <div class="alert alert-success">
                    <i class="fas fa-check-circle"></i> <?= htmlspecialchars($success) ?>
                </div>
            <?php endif; ?>

            <?php if (isset($error)): ?>
                <div class="alert alert-error">
                    <i class="fas fa-exclamation-circle"></i> <?= htmlspecialchars($error) ?>
                </div>
            <?php endif; ?>

            <!-- Statistics Cards -->
            <div class="stats-grid" style="display: grid; grid-template-columns: repeat(4, 1fr); gap: 1rem; margin-bottom: 2rem;">
                <div class="stat-card">
                    <h3>Total Orders</h3>
                    <div class="stat-value"><?= $stats['total_orders'] ?></div>
                </div>
                <div class="stat-card">
                    <h3>Pending</h3>
                    <div class="stat-value"><?= $stats['pending_orders'] ?></div>
                </div>
                <div class="stat-card">
                    <h3>Delivered</h3>
                    <div class="stat-value"><?= $stats['delivered_orders'] ?></div>
                </div>
                <div class="stat-card">
                    <h3>Revenue</h3>
                    <div class="stat-value">₹<?= number_format($stats['total_revenue']) ?></div>
                </div>
            </div>

            <!-- Filter Tabs -->
            <div class="filter-tabs">
                <?php foreach ($allowed_filters as $f): ?>
                    <a href="?status=<?= urlencode($f) ?>" 
                       class="filter-tab <?= $filter === $f ? 'active' : '' ?>">
                        <?= ucfirst($f === 'all' ? 'All Orders' : $f) ?>
                    </a>
                <?php endforeach; ?>
            </div>

            <!-- Orders Table -->
            <div class="orders-table">
                <table>
                    <thead>
                        <tr>
                            <th>Order ID</th>
                            <th>Customer</th>
                            <th>Total</th>
                            <th>Payment</th>
                            <th>Status</th>
                            <th>Date</th>
                            <th>Action</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php while ($order = $orders->fetch_assoc()): ?>
                            <tr>
                                <td><span class="order-id"><?= htmlspecialchars($order['order_id']) ?></span></td>
                                <td>
                                    <strong><?= htmlspecialchars($order['customer_name']) ?></strong><br>
                                    <small><?= htmlspecialchars($order['customer_email']) ?></small>
                                </td>
                                <td><strong>₹<?= number_format($order['total_amount']) ?></strong></td>
                                <td><?= strtoupper(htmlspecialchars($order['payment_method'])) ?></td>
                                <td>
                                    <span class="status-badge status-<?= htmlspecialchars($order['status']) ?>">
                                        <?= ucfirst(htmlspecialchars($order['status'])) ?>
                                    </span>
                                </td>
                                <td><?= date('d M Y', strtotime($order['created_at'])) ?></td>
                                <td>
                                    <button class="btn-view" onclick="viewOrder(<?= htmlspecialchars(json_encode($order), ENT_QUOTES, 'UTF-8') ?>)">
                                        <i class="fas fa-eye"></i> View
                                    </button>
                                </td>
                            </tr>
                        <?php endwhile; ?>
                    </tbody>
                </table>

                <?php if ($orders->num_rows === 0): ?>
                    <div style="text-align: center; padding: 3rem; color: #999;">
                        <i class="fas fa-shopping-cart" style="font-size: 3rem; margin-bottom: 1rem; display: block;"></i>
                        <p>No orders found</p>
                    </div>
                <?php endif; ?>
            </div>
        </main>
    </div>

    <!-- Order Detail Modal -->
    <div class="modal" id="orderModal">
        <div class="modal-content">
            <div class="modal-header">
                <h2>Order Details</h2>
                <button class="modal-close" onclick="closeModal()">&times;</button>
            </div>
            <div class="modal-body" id="orderDetails"></div>
        </div>
    </div>

    <script>
        function viewOrder(order) {
            const modal = document.getElementById('orderModal');
            const details = document.getElementById('orderDetails');
            
            // Fetch order items securely
            fetch(`get-order-items.php?order_id=${encodeURIComponent(order.order_id)}`)
                .then(res => {
                    if (!res.ok) throw new Error('Failed to fetch order items');
                    return res.json();
                })
                .then(items => {
                    let itemsHtml = '';
                    items.forEach(item => {
                        // Escape HTML to prevent XSS
                        const name = escapeHtml(item.product_name);
                        const size = escapeHtml(item.size);
                        const price = parseFloat(item.price).toLocaleString();
                        
                        itemsHtml += `
                            <div class="order-item">
                                <div style="flex: 1;">
                                    <strong>${name}</strong><br>
                                    <small style="color: #666;">Size: ${size} | Qty: ${item.quantity}</small>
                                </div>
                                <div style="font-weight: bold;">₹${price}</div>
                            </div>
                        `;
                    });

                    details.innerHTML = `
                        <div class="order-detail-section">
                            <h3><i class="fas fa-box"></i> Order Items</h3>
                            ${itemsHtml}
                        </div>

                        <div class="order-detail-section">
                            <h3><i class="fas fa-edit"></i> Update Status</h3>
                            <form method="POST">
                                <input type="hidden" name="order_id" value="${escapeHtml(order.order_id)}">
                                <input type="hidden" name="csrf_token" value="<?= generateCSRFToken() ?>">
                                <input type="hidden" name="update_status" value="1">
                                <select name="status" class="select-status">
                                    <option value="pending" ${order.status === 'pending' ? 'selected' : ''}>Pending</option>
                                    <option value="processing" ${order.status === 'processing' ? 'selected' : ''}>Processing</option>
                                    <option value="shipped" ${order.status === 'shipped' ? 'selected' : ''}>Shipped</option>
                                    <option value="delivered" ${order.status === 'delivered' ? 'selected' : ''}>Delivered</option>
                                    <option value="cancelled" ${order.status === 'cancelled' ? 'selected' : ''}>Cancelled</option>
                                </select>
                                <button type="submit" class="btn-update">
                                    <i class="fas fa-save"></i> Update Status
                                </button>
                            </form>
                        </div>
                    `;

                    modal.classList.add('active');
                })
                .catch(error => {
                    console.error('Error:', error);
                    alert('Failed to load order details');
                });
        }

        function closeModal() {
            document.getElementById('orderModal').classList.remove('active');
        }

        function escapeHtml(text) {
            const div = document.createElement('div');
            div.textContent = text;
            return div.innerHTML;
        }

        // Close modal on outside click
        document.getElementById('orderModal').addEventListener('click', function(e) {
            if (e.target === this) {
                closeModal();
            }
        });
    </script>
</body>
</html>
<?php $conn->close(); ?>