<?php
/**
 * IMAGE UPLOAD SYSTEM
 * Direct image upload for products (no links needed!)
 */

session_start();
require_once '../config/database.php';

if (!isset($_SESSION['admin_logged_in'])) {
    die(json_encode(['success' => false, 'error' => 'Not authorized']));
}

// Create uploads directory if it doesn't exist
$upload_dir = '../assets/images/products/';
if (!file_exists($upload_dir)) {
    mkdir($upload_dir, 0777, true);
}

// Handle image upload
if (isset($_FILES['image'])) {
    $file = $_FILES['image'];
    
    // Validate file
    $allowed_types = ['image/jpeg', 'image/jpg', 'image/png', 'image/gif', 'image/webp'];
    $max_size = 5 * 1024 * 1024; // 5MB
    
    if (!in_array($file['type'], $allowed_types)) {
        die(json_encode(['success' => false, 'error' => 'Invalid file type. Only JPG, PNG, GIF, WEBP allowed.']));
    }
    
    if ($file['size'] > $max_size) {
        die(json_encode(['success' => false, 'error' => 'File too large. Max 5MB allowed.']));
    }
    
    // Generate unique filename
    $extension = pathinfo($file['name'], PATHINFO_EXTENSION);
    $filename = 'product_' . time() . '_' . uniqid() . '.' . $extension;
    $filepath = $upload_dir . $filename;
    
    // Resize and compress image
    if (move_uploaded_file($file['tmp_name'], $filepath)) {
        // Optimize image
        optimizeImage($filepath, $file['type']);
        
        // Return URL for saving to database
        $image_url = 'assets/images/products/' . $filename;
        
        echo json_encode([
            'success' => true, 
            'url' => $image_url,
            'filename' => $filename
        ]);
    } else {
        echo json_encode(['success' => false, 'error' => 'Failed to upload file.']);
    }
} else {
    echo json_encode(['success' => false, 'error' => 'No file uploaded.']);
}

/**
 * Optimize and resize image
 */
function optimizeImage($filepath, $mime_type) {
    $max_width = 800;
    $max_height = 800;
    $quality = 85;
    
    list($width, $height) = getimagesize($filepath);
    
    // Calculate new dimensions
    $ratio = min($max_width / $width, $max_height / $height);
    if ($ratio < 1) {
        $new_width = $width * $ratio;
        $new_height = $height * $ratio;
    } else {
        $new_width = $width;
        $new_height = $height;
    }
    
    // Create image based on type
    switch ($mime_type) {
        case 'image/jpeg':
        case 'image/jpg':
            $source = imagecreatefromjpeg($filepath);
            break;
        case 'image/png':
            $source = imagecreatefrompng($filepath);
            break;
        case 'image/gif':
            $source = imagecreatefromgif($filepath);
            break;
        case 'image/webp':
            $source = imagecreatefromwebp($filepath);
            break;
        default:
            return;
    }
    
    // Create new image
    $thumb = imagecreatetruecolor($new_width, $new_height);
    
    // Preserve transparency for PNG and GIF
    if ($mime_type === 'image/png' || $mime_type === 'image/gif') {
        imagealphablending($thumb, false);
        imagesavealpha($thumb, true);
    }
    
    imagecopyresampled($thumb, $source, 0, 0, 0, 0, $new_width, $new_height, $width, $height);
    
    // Save optimized image
    switch ($mime_type) {
        case 'image/jpeg':
        case 'image/jpg':
            imagejpeg($thumb, $filepath, $quality);
            break;
        case 'image/png':
            imagepng($thumb, $filepath, 9);
            break;
        case 'image/gif':
            imagegif($thumb, $filepath);
            break;
        case 'image/webp':
            imagewebp($thumb, $filepath, $quality);
            break;
    }
    
    imagedestroy($source);
    imagedestroy($thumb);
}
?>

<!-- UPDATED PRODUCTS.PHP WITH IMAGE UPLOAD -->

<?php
/**
 * Add this to admin/products.php
 * Replace the image URL input with this upload widget
 */
?>

<style>
.image-upload-container {
    border: 2px dashed #667eea;
    border-radius: 12px;
    padding: 2rem;
    text-align: center;
    cursor: pointer;
    transition: all 0.3s;
}

.image-upload-container:hover {
    background: #f8f9ff;
    border-color: #764ba2;
}

.image-preview {
    max-width: 200px;
    max-height: 200px;
    margin: 1rem auto;
    border-radius: 8px;
}

.upload-text {
    color: #667eea;
    font-weight: 600;
}

.uploading {
    opacity: 0.6;
    pointer-events: none;
}
</style>

<div class="form-group">
    <label>Product Image *</label>
    
    <!-- Hidden file input -->
    <input type="file" id="imageInput" accept="image/*" style="display: none;">
    <input type="hidden" name="image_url" id="imageUrl" required>
    
    <!-- Upload area -->
    <div class="image-upload-container" id="uploadArea" onclick="document.getElementById('imageInput').click()">
        <i class="fas fa-cloud-upload-alt" style="font-size: 3rem; color: #667eea; margin-bottom: 1rem;"></i>
        <div class="upload-text">Click to upload image</div>
        <div style="color: #666; font-size: 0.9rem; margin-top: 0.5rem;">
            JPG, PNG, GIF or WEBP (Max 5MB)
        </div>
        <img id="imagePreview" class="image-preview" style="display: none;">
    </div>
    
    <div id="uploadStatus" style="margin-top: 1rem; text-align: center;"></div>
</div>

<script>
document.getElementById('imageInput').addEventListener('change', function(e) {
    const file = e.target.files[0];
    if (!file) return;
    
    // Show preview
    const reader = new FileReader();
    reader.onload = function(e) {
        const preview = document.getElementById('imagePreview');
        preview.src = e.target.result;
        preview.style.display = 'block';
    };
    reader.readAsDataURL(file);
    
    // Upload file
    uploadImage(file);
});

function uploadImage(file) {
    const uploadArea = document.getElementById('uploadArea');
    const uploadStatus = document.getElementById('uploadStatus');
    
    uploadArea.classList.add('uploading');
    uploadStatus.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Uploading...';
    
    const formData = new FormData();
    formData.append('image', file);
    
    fetch('upload-image.php', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            document.getElementById('imageUrl').value = data.url;
            uploadStatus.innerHTML = '<span style="color: #2ecc71;"><i class="fas fa-check-circle"></i> Image uploaded successfully!</span>';
        } else {
            uploadStatus.innerHTML = '<span style="color: #e74c3c;"><i class="fas fa-times-circle"></i> ' + data.error + '</span>';
        }
        uploadArea.classList.remove('uploading');
    })
    .catch(error => {
        uploadStatus.innerHTML = '<span style="color: #e74c3c;"><i class="fas fa-times-circle"></i> Upload failed. Try again.</span>';
        uploadArea.classList.remove('uploading');
    });
}
</script>

<!-- CREATE ASSETS FOLDER STRUCTURE -->
<?php
/**
 * Run this once to create folder structure
 * Save as: setup-folders.php and run once
 */

$folders = [
    '../assets',
    '../assets/images',
    '../assets/images/products',
    '../assets/images/uploads'
];

foreach ($folders as $folder) {
    if (!file_exists($folder)) {
        mkdir($folder, 0777, true);
        echo "Created: $folder<br>";
    }
}

// Create .htaccess to allow image access
$htaccess = '../assets/.htaccess';
if (!file_exists($htaccess)) {
    file_put_contents($htaccess, "Options +Indexes\nallow from all");
}

echo "Folders created successfully!";
?>

<!-- UPDATE DATABASE.SQL -->
/*
No database changes needed for image upload!
Just stores the local path instead of external URL
*/