<?php
/**
 * PRODUCT REVIEWS SYSTEM - COMPLETE IMPLEMENTATION
 * 
 * Features:
 * - Add reviews with ratings
 * - Display reviews on product page
 * - Average rating calculation
 * - Verified purchase badge
 * - Review moderation
 */

session_start();
require_once '../config/database.php';

// Handle review submission
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['submit_review'])) {
    $conn = getConnection();
    
    $product_id = $_POST['product_id'];
    $customer_name = $_POST['customer_name'];
    $customer_email = $_POST['customer_email'];
    $rating = $_POST['rating'];
    $review_title = $_POST['review_title'];
    $review_text = $_POST['review_text'];
    
    // Check if verified purchase
    $is_verified = false;
    if (isset($_SESSION['customer_id'])) {
        $check = $conn->query("SELECT COUNT(*) as count FROM order_items oi 
                              JOIN orders o ON oi.order_id = o.order_id 
                              WHERE oi.product_id = $product_id 
                              AND o.customer_email = '$customer_email' 
                              AND o.status = 'delivered'");
        $is_verified = $check->fetch_assoc()['count'] > 0;
    }
    
    // Insert review
    $stmt = $conn->prepare("INSERT INTO product_reviews (product_id, customer_name, customer_email, rating, review_title, review_text, is_verified_purchase) VALUES (?, ?, ?, ?, ?, ?, ?)");
    $stmt->bind_param("isssssi", $product_id, $customer_name, $customer_email, $rating, $review_title, $review_text, $is_verified);
    
    if ($stmt->execute()) {
        // Update product average rating
        updateProductRating($conn, $product_id);
        $success = "Review submitted successfully!";
    } else {
        $error = "Failed to submit review. Please try again.";
    }
    
    $conn->close();
}

function updateProductRating($conn, $product_id) {
    // Calculate average rating
    $result = $conn->query("SELECT AVG(rating) as avg_rating, COUNT(*) as count 
                            FROM product_reviews 
                            WHERE product_id = $product_id AND is_approved = 1");
    $data = $result->fetch_assoc();
    
    $avg_rating = round($data['avg_rating'], 1);
    $count = $data['count'];
    
    // Update product
    $conn->query("UPDATE products SET average_rating = $avg_rating, reviews_count = $count WHERE product_id = $product_id");
}

function getProductReviews($conn, $product_id) {
    return $conn->query("SELECT * FROM product_reviews 
                        WHERE product_id = $product_id AND is_approved = 1 
                        ORDER BY created_at DESC");
}

function displayStarRating($rating) {
    $html = '';
    for ($i = 1; $i <= 5; $i++) {
        if ($i <= $rating) {
            $html .= '<i class="fas fa-star" style="color: #ffc107;"></i>';
        } else {
            $html .= '<i class="far fa-star" style="color: #ffc107;"></i>';
        }
    }
    return $html;
}

// Get product ID from URL
$product_id = $_GET['product_id'] ?? 1;
$conn = getConnection();
$product = $conn->query("SELECT * FROM products WHERE product_id = $product_id")->fetch_assoc();
$reviews = getProductReviews($conn, $product_id);
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?= htmlspecialchars($product['name']) ?> - Reviews</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Oxygen, Ubuntu, Cantarell, sans-serif;
            background: #f8f9fa;
        }

        .container {
            max-width: 1200px;
            margin: 2rem auto;
            padding: 0 1rem;
        }

        .product-header {
            background: white;
            padding: 2rem;
            border-radius: 15px;
            box-shadow: 0 4px 15px rgba(0,0,0,0.1);
            margin-bottom: 2rem;
            display: flex;
            gap: 2rem;
            align-items: center;
        }

        .product-image {
            width: 120px;
            height: 120px;
            object-fit: cover;
            border-radius: 12px;
        }

        .product-info h1 {
            font-size: 1.8rem;
            margin-bottom: 0.5rem;
        }

        .rating-summary {
            background: white;
            padding: 2rem;
            border-radius: 15px;
            box-shadow: 0 4px 15px rgba(0,0,0,0.1);
            margin-bottom: 2rem;
        }

        .rating-overview {
            display: flex;
            gap: 3rem;
            align-items: center;
        }

        .rating-score {
            text-align: center;
        }

        .rating-number {
            font-size: 4rem;
            font-weight: bold;
            color: #667eea;
        }

        .rating-stars {
            font-size: 1.5rem;
            margin: 0.5rem 0;
        }

        .rating-count {
            color: #666;
        }

        .rating-bars {
            flex: 1;
        }

        .rating-bar {
            display: flex;
            align-items: center;
            gap: 1rem;
            margin-bottom: 0.75rem;
        }

        .rating-bar-fill {
            flex: 1;
            height: 8px;
            background: #e0e0e0;
            border-radius: 10px;
            overflow: hidden;
        }

        .rating-bar-inner {
            height: 100%;
            background: #ffc107;
        }

        .review-form {
            background: white;
            padding: 2rem;
            border-radius: 15px;
            box-shadow: 0 4px 15px rgba(0,0,0,0.1);
            margin-bottom: 2rem;
        }

        .form-group {
            margin-bottom: 1.5rem;
        }

        .form-group label {
            display: block;
            margin-bottom: 0.5rem;
            font-weight: 600;
        }

        .form-control {
            width: 100%;
            padding: 0.75rem;
            border: 2px solid #e0e0e0;
            border-radius: 10px;
            font-size: 1rem;
        }

        .form-control:focus {
            outline: none;
            border-color: #667eea;
        }

        .star-rating {
            display: flex;
            gap: 0.5rem;
            font-size: 2rem;
        }

        .star-rating i {
            cursor: pointer;
            color: #ddd;
            transition: color 0.3s;
        }

        .star-rating i:hover,
        .star-rating i.active {
            color: #ffc107;
        }

        .btn-submit {
            padding: 1rem 2rem;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            border: none;
            border-radius: 10px;
            font-weight: bold;
            cursor: pointer;
        }

        .reviews-list {
            background: white;
            padding: 2rem;
            border-radius: 15px;
            box-shadow: 0 4px 15px rgba(0,0,0,0.1);
        }

        .review-item {
            padding: 1.5rem;
            border-bottom: 1px solid #e0e0e0;
        }

        .review-item:last-child {
            border-bottom: none;
        }

        .review-header {
            display: flex;
            justify-content: space-between;
            align-items: start;
            margin-bottom: 1rem;
        }

        .reviewer-info {
            display: flex;
            align-items: center;
            gap: 1rem;
        }

        .reviewer-avatar {
            width: 50px;
            height: 50px;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            font-weight: bold;
            font-size: 1.2rem;
        }

        .reviewer-name {
            font-weight: 600;
            margin-bottom: 0.25rem;
        }

        .review-date {
            color: #666;
            font-size: 0.85rem;
        }

        .verified-badge {
            background: #4caf50;
            color: white;
            padding: 0.25rem 0.75rem;
            border-radius: 20px;
            font-size: 0.75rem;
            font-weight: 600;
        }

        .review-title {
            font-weight: 600;
            font-size: 1.1rem;
            margin-bottom: 0.5rem;
        }

        .review-text {
            color: #666;
            line-height: 1.6;
        }

        .alert {
            padding: 1rem;
            border-radius: 10px;
            margin-bottom: 1rem;
        }

        .alert-success {
            background: #d4edda;
            color: #155724;
        }

        .alert-error {
            background: #f8d7da;
            color: #721c24;
        }
    </style>
</head>
<body>
    <div class="container">
        <!-- Product Header -->
        <div class="product-header">
            <img src="<?= htmlspecialchars($product['image_url']) ?>" alt="<?= htmlspecialchars($product['name']) ?>" class="product-image">
            <div class="product-info">
                <h1><?= htmlspecialchars($product['name']) ?></h1>
                <div class="rating-stars">
                    <?= displayStarRating($product['average_rating']) ?>
                    <span style="color: #666; margin-left: 0.5rem;">
                        <?= $product['average_rating'] ?> (<?= $product['reviews_count'] ?> reviews)
                    </span>
                </div>
            </div>
        </div>

        <!-- Rating Summary -->
        <div class="rating-summary">
            <h2 style="margin-bottom: 1.5rem;">Customer Reviews</h2>
            <div class="rating-overview">
                <div class="rating-score">
                    <div class="rating-number"><?= $product['average_rating'] ?></div>
                    <div class="rating-stars"><?= displayStarRating($product['average_rating']) ?></div>
                    <div class="rating-count"><?= $product['reviews_count'] ?> ratings</div>
                </div>

                <div class="rating-bars">
                    <?php
                    for ($i = 5; $i >= 1; $i--) {
                        $count = $conn->query("SELECT COUNT(*) as count FROM product_reviews WHERE product_id = $product_id AND rating = $i AND is_approved = 1")->fetch_assoc()['count'];
                        $percentage = $product['reviews_count'] > 0 ? ($count / $product['reviews_count']) * 100 : 0;
                    ?>
                        <div class="rating-bar">
                            <span style="width: 60px;"><?= $i ?> <i class="fas fa-star" style="color: #ffc107; font-size: 0.8rem;"></i></span>
                            <div class="rating-bar-fill">
                                <div class="rating-bar-inner" style="width: <?= $percentage ?>%"></div>
                            </div>
                            <span style="width: 50px; text-align: right; color: #666;"><?= $count ?></span>
                        </div>
                    <?php } ?>
                </div>
            </div>
        </div>

        <!-- Review Form -->
        <div class="review-form">
            <h2 style="margin-bottom: 1.5rem;">Write a Review</h2>

            <?php if (isset($success)): ?>
                <div class="alert alert-success">
                    <i class="fas fa-check-circle"></i> <?= $success ?>
                </div>
            <?php endif; ?>

            <?php if (isset($error)): ?>
                <div class="alert alert-error">
                    <i class="fas fa-exclamation-circle"></i> <?= $error ?>
                </div>
            <?php endif; ?>

            <form method="POST">
                <input type="hidden" name="submit_review" value="1">
                <input type="hidden" name="product_id" value="<?= $product_id ?>">
                <input type="hidden" name="rating" id="ratingInput" value="5">

                <div class="form-group">
                    <label>Your Rating *</label>
                    <div class="star-rating" id="starRating">
                        <i class="fas fa-star active" data-rating="1"></i>
                        <i class="fas fa-star active" data-rating="2"></i>
                        <i class="fas fa-star active" data-rating="3"></i>
                        <i class="fas fa-star active" data-rating="4"></i>
                        <i class="fas fa-star active" data-rating="5"></i>
                    </div>
                </div>

                <div class="form-group">
                    <label>Your Name *</label>
                    <input type="text" name="customer_name" class="form-control" required>
                </div>

                <div class="form-group">
                    <label>Your Email *</label>
                    <input type="email" name="customer_email" class="form-control" required>
                </div>

                <div class="form-group">
                    <label>Review Title</label>
                    <input type="text" name="review_title" class="form-control" placeholder="e.g., Great product!">
                </div>

                <div class="form-group">
                    <label>Your Review *</label>
                    <textarea name="review_text" class="form-control" rows="5" placeholder="Share your experience with this product..." required></textarea>
                </div>

                <button type="submit" class="btn-submit">
                    <i class="fas fa-paper-plane"></i> Submit Review
                </button>
            </form>
        </div>

        <!-- Reviews List -->
        <div class="reviews-list">
            <h2 style="margin-bottom: 1.5rem;">All Reviews (<?= $product['reviews_count'] ?>)</h2>

            <?php if ($reviews->num_rows === 0): ?>
                <p style="text-align: center; color: #666; padding: 3rem;">
                    No reviews yet. Be the first to review this product!
                </p>
            <?php else: ?>
                <?php while ($review = $reviews->fetch_assoc()): ?>
                    <div class="review-item">
                        <div class="review-header">
                            <div class="reviewer-info">
                                <div class="reviewer-avatar">
                                    <?= strtoupper(substr($review['customer_name'], 0, 1)) ?>
                                </div>
                                <div>
                                    <div class="reviewer-name"><?= htmlspecialchars($review['customer_name']) ?></div>
                                    <div class="review-date">
                                        <?= date('F d, Y', strtotime($review['created_at'])) ?>
                                    </div>
                                </div>
                            </div>
                            <div style="text-align: right;">
                                <div style="margin-bottom: 0.5rem;">
                                    <?= displayStarRating($review['rating']) ?>
                                </div>
                                <?php if ($review['is_verified_purchase']): ?>
                                    <span class="verified-badge">
                                        <i class="fas fa-check"></i> Verified Purchase
                                    </span>
                                <?php endif; ?>
                            </div>
                        </div>

                        <?php if ($review['review_title']): ?>
                            <div class="review-title"><?= htmlspecialchars($review['review_title']) ?></div>
                        <?php endif; ?>

                        <div class="review-text"><?= nl2br(htmlspecialchars($review['review_text'])) ?></div>
                    </div>
                <?php endwhile; ?>
            <?php endif; ?>
        </div>
    </div>

    <script>
        // Star rating interaction
        const stars = document.querySelectorAll('#starRating i');
        const ratingInput = document.getElementById('ratingInput');

        stars.forEach(star => {
            star.addEventListener('click', function() {
                const rating = this.getAttribute('data-rating');
                ratingInput.value = rating;

                // Update visual state
                stars.forEach((s, index) => {
                    if (index < rating) {
                        s.classList.add('active');
                    } else {
                        s.classList.remove('active');
                    }
                });
            });

            star.addEventListener('mouseenter', function() {
                const rating = this.getAttribute('data-rating');
                stars.forEach((s, index) => {
                    if (index < rating) {
                        s.style.color = '#ffc107';
                    } else {
                        s.style.color = '#ddd';
                    }
                });
            });
        });

        document.getElementById('starRating').addEventListener('mouseleave', function() {
            const currentRating = ratingInput.value;
            stars.forEach((s, index) => {
                if (index < currentRating) {
                    s.style.color = '#ffc107';
                } else {
                    s.style.color = '#ddd';
                }
            });
        });
    </script>
</body>
</html>
<?php $conn->close(); ?>