<?php
/**
 * RAZORPAY PAYMENT GATEWAY INTEGRATION
 * 
 * Setup Instructions:
 * 1. Sign up at https://razorpay.com
 * 2. Get API Keys from Dashboard
 * 3. Add keys to config below
 * 4. Install Razorpay PHP SDK (optional) or use this direct integration
 */

session_start();
require_once '../config/database.php';

// Razorpay Configuration
define('RAZORPAY_KEY_ID', 'rzp_test_YOUR_KEY_ID'); // Get from Razorpay Dashboard
define('RAZORPAY_KEY_SECRET', 'YOUR_KEY_SECRET');    // Keep this secret!

$conn = getConnection();

// Get cart and calculate total
$cart = json_decode($_POST['cart'] ?? '[]', true);
$total_amount = $_POST['total_amount'] ?? 0;
$customer_data = [
    'name' => $_POST['name'] ?? '',
    'email' => $_POST['email'] ?? '',
    'phone' => $_POST['phone'] ?? '',
    'address' => $_POST['address'] ?? '',
    'city' => $_POST['city'] ?? '',
    'state' => $_POST['state'] ?? '',
    'pincode' => $_POST['pincode'] ?? ''
];

// Generate Order ID
$order_id = 'ORD' . time();
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Payment - Boutique</title>
    <script src="https://checkout.razorpay.com/v1/checkout.js"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Oxygen, Ubuntu, Cantarell, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            display: flex;
            align-items: center;
            justify-content: center;
            padding: 2rem;
        }

        .payment-container {
            background: white;
            border-radius: 20px;
            box-shadow: 0 20px 60px rgba(0,0,0,0.3);
            max-width: 500px;
            width: 100%;
            overflow: hidden;
        }

        .payment-header {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 2rem;
            text-align: center;
        }

        .payment-header h1 {
            font-size: 1.8rem;
            margin-bottom: 0.5rem;
        }

        .payment-body {
            padding: 2rem;
        }

        .amount-box {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 2rem;
            border-radius: 15px;
            text-align: center;
            margin-bottom: 2rem;
        }

        .amount-label {
            font-size: 0.9rem;
            opacity: 0.9;
            margin-bottom: 0.5rem;
        }

        .amount-value {
            font-size: 3rem;
            font-weight: bold;
        }

        .payment-methods {
            margin-bottom: 2rem;
        }

        .payment-method {
            padding: 1.5rem;
            border: 2px solid #e0e0e0;
            border-radius: 12px;
            margin-bottom: 1rem;
            cursor: pointer;
            transition: all 0.3s;
            display: flex;
            align-items: center;
            gap: 1rem;
        }

        .payment-method:hover {
            border-color: #667eea;
            background: #f8f9ff;
        }

        .payment-method input[type="radio"] {
            width: 20px;
            height: 20px;
            cursor: pointer;
        }

        .payment-method.selected {
            border-color: #667eea;
            background: #f0f4ff;
        }

        .payment-icon {
            width: 40px;
            height: 40px;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            border-radius: 10px;
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            font-size: 1.2rem;
        }

        .btn-pay {
            width: 100%;
            padding: 1.25rem;
            background: linear-gradient(135deg, #2ecc71 0%, #27ae60 100%);
            color: white;
            border: none;
            border-radius: 12px;
            font-size: 1.1rem;
            font-weight: bold;
            cursor: pointer;
            transition: all 0.3s;
        }

        .btn-pay:hover {
            transform: translateY(-2px);
            box-shadow: 0 8px 25px rgba(46, 204, 113, 0.4);
        }

        .btn-pay:disabled {
            background: #ccc;
            cursor: not-allowed;
            transform: none;
        }

        .security-badge {
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 0.5rem;
            color: #2ecc71;
            margin-top: 1rem;
            font-size: 0.9rem;
        }

        .order-summary {
            background: #f8f9fa;
            padding: 1.5rem;
            border-radius: 12px;
            margin-bottom: 2rem;
        }

        .summary-row {
            display: flex;
            justify-content: space-between;
            margin-bottom: 0.75rem;
            color: #666;
        }
    </style>
</head>
<body>
    <div class="payment-container">
        <div class="payment-header">
            <h1><i class="fas fa-credit-card"></i> Secure Payment</h1>
            <p>Complete your order</p>
        </div>

        <div class="payment-body">
            <div class="amount-box">
                <div class="amount-label">Total Amount</div>
                <div class="amount-value">₹<?= number_format($total_amount) ?></div>
            </div>

            <div class="order-summary">
                <h3 style="margin-bottom: 1rem; color: #333;">Order Summary</h3>
                <div class="summary-row">
                    <span>Order ID:</span>
                    <strong><?= $order_id ?></strong>
                </div>
                <div class="summary-row">
                    <span>Name:</span>
                    <strong><?= htmlspecialchars($customer_data['name']) ?></strong>
                </div>
                <div class="summary-row">
                    <span>Email:</span>
                    <strong><?= htmlspecialchars($customer_data['email']) ?></strong>
                </div>
            </div>

            <div class="payment-methods">
                <h3 style="margin-bottom: 1rem; color: #333;">Select Payment Method</h3>
                
                <label class="payment-method" onclick="selectPayment('razorpay')">
                    <input type="radio" name="payment_method" value="razorpay" checked>
                    <div class="payment-icon">
                        <i class="fas fa-credit-card"></i>
                    </div>
                    <div>
                        <strong>Pay Online</strong>
                        <div style="font-size: 0.85rem; color: #666;">UPI, Cards, Net Banking, Wallets</div>
                    </div>
                </label>

                <label class="payment-method" onclick="selectPayment('cod')">
                    <input type="radio" name="payment_method" value="cod">
                    <div class="payment-icon">
                        <i class="fas fa-money-bill-wave"></i>
                    </div>
                    <div>
                        <strong>Cash on Delivery</strong>
                        <div style="font-size: 0.85rem; color: #666;">Pay when you receive</div>
                    </div>
                </label>
            </div>

            <button class="btn-pay" id="payButton" onclick="processPayment()">
                <i class="fas fa-lock"></i> Proceed to Pay ₹<?= number_format($total_amount) ?>
            </button>

            <div class="security-badge">
                <i class="fas fa-shield-alt"></i>
                <span>Secure Payment - 256-bit SSL Encrypted</span>
            </div>
        </div>
    </div>

    <script>
        let selectedPaymentMethod = 'razorpay';
        const cartData = <?= json_encode($cart) ?>;
        const customerData = <?= json_encode($customer_data) ?>;
        const totalAmount = <?= $total_amount ?>;
        const orderId = '<?= $order_id ?>';

        function selectPayment(method) {
            selectedPaymentMethod = method;
            document.querySelectorAll('.payment-method').forEach(el => {
                el.classList.remove('selected');
            });
            event.currentTarget.classList.add('selected');
            
            const btn = document.getElementById('payButton');
            if (method === 'cod') {
                btn.innerHTML = '<i class="fas fa-check-circle"></i> Place Order (COD)';
            } else {
                btn.innerHTML = '<i class="fas fa-lock"></i> Proceed to Pay ₹' + totalAmount.toLocaleString();
            }
        }

        function processPayment() {
            if (selectedPaymentMethod === 'razorpay') {
                openRazorpay();
            } else {
                placeCODOrder();
            }
        }

        function openRazorpay() {
            const options = {
                key: '<?= RAZORPAY_KEY_ID ?>',
                amount: totalAmount * 100, // Razorpay expects amount in paise
                currency: 'INR',
                name: 'Boutique Store',
                description: 'Order #' + orderId,
                image: 'https://yourdomain.com/logo.png',
                order_id: '', // Generate from backend for better security
                handler: function (response) {
                    // Payment successful
                    verifyPayment(response);
                },
                prefill: {
                    name: customerData.name,
                    email: customerData.email,
                    contact: customerData.phone
                },
                notes: {
                    order_id: orderId
                },
                theme: {
                    color: '#667eea'
                },
                modal: {
                    ondismiss: function() {
                        alert('Payment cancelled. Please try again.');
                    }
                }
            };

            const razorpay = new Razorpay(options);
            razorpay.open();
        }

        function verifyPayment(response) {
            // Send payment details to server for verification
            const formData = new FormData();
            formData.append('razorpay_payment_id', response.razorpay_payment_id);
            formData.append('razorpay_order_id', response.razorpay_order_id);
            formData.append('razorpay_signature', response.razorpay_signature);
            formData.append('order_id', orderId);
            formData.append('cart', JSON.stringify(cartData));
            formData.append('customer_data', JSON.stringify(customerData));
            formData.append('total_amount', totalAmount);

            fetch('verify-payment.php', {
                method: 'POST',
                body: formData
            })
            .then(res => res.json())
            .then(data => {
                if (data.success) {
                    localStorage.removeItem('cart');
                    window.location.href = 'order-confirmation.php?order_id=' + orderId;
                } else {
                    alert('Payment verification failed. Please contact support.');
                }
            })
            .catch(error => {
                console.error('Error:', error);
                alert('An error occurred. Please try again.');
            });
        }

        function placeCODOrder() {
            const formData = new FormData();
            formData.append('order_id', orderId);
            formData.append('cart', JSON.stringify(cartData));
            formData.append('customer_data', JSON.stringify(customerData));
            formData.append('total_amount', totalAmount);
            formData.append('payment_method', 'cod');

            fetch('place-order.php', {
                method: 'POST',
                body: formData
            })
            .then(res => res.json())
            .then(data => {
                if (data.success) {
                    localStorage.removeItem('cart');
                    window.location.href = 'order-confirmation.php?order_id=' + orderId;
                } else {
                    alert('Order placement failed. Please try again.');
                }
            })
            .catch(error => {
                console.error('Error:', error);
                alert('An error occurred. Please try again.');
            });
        }
    </script>
</body>
</html>