<?php
/**
 * EMAIL NOTIFICATION SYSTEM
 * Handles all email notifications for the boutique
 */

class EmailService {
    
    private $from_email = 'noreply@yourboutique.com';
    private $from_name = 'Boutique Store';
    
    /**
     * Send Order Confirmation Email
     */
    public function sendOrderConfirmation($order, $items) {
        $to = $order['customer_email'];
        $subject = "Order Confirmation - " . $order['order_id'];
        
        $message = $this->getOrderConfirmationTemplate($order, $items);
        
        return $this->sendEmail($to, $subject, $message);
    }
    
    /**
     * Send Order Status Update Email
     */
    public function sendStatusUpdate($order, $new_status) {
        $to = $order['customer_email'];
        $subject = "Order Update - " . $order['order_id'];
        
        $message = $this->getStatusUpdateTemplate($order, $new_status);
        
        return $this->sendEmail($to, $subject, $message);
    }
    
    /**
     * Send Welcome Email
     */
    public function sendWelcomeEmail($email, $name) {
        $subject = "Welcome to Our Boutique!";
        $message = $this->getWelcomeTemplate($name);
        
        return $this->sendEmail($email, $subject, $message);
    }
    
    /**
     * Core Email Sending Function
     */
    private function sendEmail($to, $subject, $message) {
        $headers = "MIME-Version: 1.0" . "\r\n";
        $headers .= "Content-type:text/html;charset=UTF-8" . "\r\n";
        $headers .= "From: {$this->from_name} <{$this->from_email}>" . "\r\n";
        
        return mail($to, $subject, $message, $headers);
    }
    
    /**
     * Order Confirmation Email Template
     */
    private function getOrderConfirmationTemplate($order, $items) {
        $itemsHtml = '';
        $subtotal = 0;
        
        foreach ($items as $item) {
            $itemTotal = $item['price'] * $item['quantity'];
            $subtotal += $itemTotal;
            $itemsHtml .= "
                <tr>
                    <td style='padding: 15px; border-bottom: 1px solid #eee;'>
                        <strong>{$item['product_name']}</strong><br>
                        <small style='color: #666;'>Size: {$item['size']} | Qty: {$item['quantity']}</small>
                    </td>
                    <td style='padding: 15px; border-bottom: 1px solid #eee; text-align: right;'>
                        ₹" . number_format($itemTotal) . "
                    </td>
                </tr>
            ";
        }
        
        return "
        <!DOCTYPE html>
        <html>
        <head>
            <meta charset='UTF-8'>
            <meta name='viewport' content='width=device-width, initial-scale=1.0'>
        </head>
        <body style='margin: 0; padding: 0; font-family: Arial, sans-serif; background-color: #f8f9fa;'>
            <table width='100%' cellpadding='0' cellspacing='0' style='background-color: #f8f9fa; padding: 20px;'>
                <tr>
                    <td align='center'>
                        <table width='600' cellpadding='0' cellspacing='0' style='background-color: white; border-radius: 10px; overflow: hidden; box-shadow: 0 4px 15px rgba(0,0,0,0.1);'>
                            
                            <!-- Header -->
                            <tr>
                                <td style='background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); padding: 40px; text-align: center; color: white;'>
                                    <h1 style='margin: 0; font-size: 32px;'>✓ Order Confirmed!</h1>
                                    <p style='margin: 10px 0 0 0; font-size: 16px; opacity: 0.9;'>Thank you for your purchase</p>
                                </td>
                            </tr>
                            
                            <!-- Order ID -->
                            <tr>
                                <td style='padding: 30px; text-align: center; background: #f8f9fa;'>
                                    <p style='margin: 0 0 10px 0; font-size: 14px; color: #666;'>Order ID</p>
                                    <h2 style='margin: 0; font-size: 24px; color: #333; font-family: monospace; letter-spacing: 2px;'>
                                        {$order['order_id']}
                                    </h2>
                                </td>
                            </tr>
                            
                            <!-- Order Items -->
                            <tr>
                                <td style='padding: 30px;'>
                                    <h3 style='margin: 0 0 20px 0; color: #333;'>Order Details</h3>
                                    <table width='100%' cellpadding='0' cellspacing='0'>
                                        {$itemsHtml}
                                        <tr>
                                            <td style='padding: 15px; text-align: right;' colspan='2'>
                                                <strong style='font-size: 18px;'>Total: ₹" . number_format($order['total_amount']) . "</strong>
                                            </td>
                                        </tr>
                                    </table>
                                </td>
                            </tr>
                            
                            <!-- Shipping Address -->
                            <tr>
                                <td style='padding: 30px; background: #f8f9fa;'>
                                    <h3 style='margin: 0 0 15px 0; color: #333;'>Shipping Address</h3>
                                    <p style='margin: 0; line-height: 1.6; color: #666;'>
                                        <strong>{$order['customer_name']}</strong><br>
                                        {$order['shipping_address']}<br>
                                        {$order['city']}, {$order['state']} - {$order['pincode']}<br>
                                        Phone: {$order['customer_phone']}
                                    </p>
                                </td>
                            </tr>
                            
                            <!-- Payment Info -->
                            <tr>
                                <td style='padding: 30px;'>
                                    <h3 style='margin: 0 0 15px 0; color: #333;'>Payment Information</h3>
                                    <p style='margin: 0; color: #666;'>
                                        <strong>Payment Method:</strong> " . strtoupper($order['payment_method']) . "<br>
                                        <strong>Payment Status:</strong> " . ($order['payment_method'] === 'cod' ? 'Pay on Delivery' : 'Paid') . "
                                    </p>
                                </td>
                            </tr>
                            
                            <!-- Delivery Info -->
                            <tr>
                                <td style='padding: 30px; background: #e3f2fd; text-align: center;'>
                                    <p style='margin: 0; color: #1976d2;'>
                                        <strong>🚚 Expected Delivery: 5-7 Business Days</strong>
                                    </p>
                                </td>
                            </tr>
                            
                            <!-- Footer -->
                            <tr>
                                <td style='padding: 30px; text-align: center; background: #f8f9fa; border-top: 1px solid #e0e0e0;'>
                                    <p style='margin: 0 0 10px 0; color: #666;'>
                                        Need help? Contact us at <a href='mailto:support@yourboutique.com' style='color: #667eea;'>support@yourboutique.com</a>
                                    </p>
                                    <p style='margin: 0; color: #999; font-size: 12px;'>
                                        © 2024 Boutique Store. All rights reserved.
                                    </p>
                                </td>
                            </tr>
                            
                        </table>
                    </td>
                </tr>
            </table>
        </body>
        </html>
        ";
    }
    
    /**
     * Status Update Email Template
     */
    private function getStatusUpdateTemplate($order, $new_status) {
        $statusMessages = [
            'processing' => ['title' => 'Order is Being Processed', 'icon' => '⚙️', 'message' => 'We are preparing your order for shipment.'],
            'shipped' => ['title' => 'Order Shipped!', 'icon' => '🚚', 'message' => 'Your order is on its way to you.'],
            'delivered' => ['title' => 'Order Delivered', 'icon' => '✓', 'message' => 'Your order has been delivered successfully.'],
            'cancelled' => ['title' => 'Order Cancelled', 'icon' => '✗', 'message' => 'Your order has been cancelled.']
        ];
        
        $status = $statusMessages[$new_status] ?? ['title' => 'Order Updated', 'icon' => 'ℹ️', 'message' => 'Your order status has been updated.'];
        
        return "
        <!DOCTYPE html>
        <html>
        <head>
            <meta charset='UTF-8'>
            <meta name='viewport' content='width=device-width, initial-scale=1.0'>
        </head>
        <body style='margin: 0; padding: 0; font-family: Arial, sans-serif; background-color: #f8f9fa;'>
            <table width='100%' cellpadding='0' cellspacing='0' style='background-color: #f8f9fa; padding: 20px;'>
                <tr>
                    <td align='center'>
                        <table width='600' cellpadding='0' cellspacing='0' style='background-color: white; border-radius: 10px; overflow: hidden;'>
                            
                            <tr>
                                <td style='background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); padding: 40px; text-align: center; color: white;'>
                                    <h1 style='margin: 0; font-size: 48px;'>{$status['icon']}</h1>
                                    <h2 style='margin: 20px 0 0 0; font-size: 28px;'>{$status['title']}</h2>
                                </td>
                            </tr>
                            
                            <tr>
                                <td style='padding: 40px; text-align: center;'>
                                    <p style='margin: 0 0 20px 0; font-size: 16px; color: #666; line-height: 1.6;'>
                                        {$status['message']}
                                    </p>
                                    <p style='margin: 0; font-size: 14px; color: #999;'>
                                        Order ID: <strong style='color: #333;'>{$order['order_id']}</strong>
                                    </p>
                                </td>
                            </tr>
                            
                            <tr>
                                <td style='padding: 30px; text-align: center; background: #f8f9fa;'>
                                    <a href='https://yourdomain.com' style='display: inline-block; padding: 15px 40px; background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; text-decoration: none; border-radius: 8px; font-weight: bold;'>
                                        View Order Details
                                    </a>
                                </td>
                            </tr>
                            
                        </table>
                    </td>
                </tr>
            </table>
        </body>
        </html>
        ";
    }
    
    /**
     * Welcome Email Template
     */
    private function getWelcomeTemplate($name) {
        return "
        <!DOCTYPE html>
        <html>
        <head>
            <meta charset='UTF-8'>
            <meta name='viewport' content='width=device-width, initial-scale=1.0'>
        </head>
        <body style='margin: 0; padding: 0; font-family: Arial, sans-serif; background-color: #f8f9fa;'>
            <table width='100%' cellpadding='0' cellspacing='0' style='background-color: #f8f9fa; padding: 20px;'>
                <tr>
                    <td align='center'>
                        <table width='600' cellpadding='0' cellspacing='0' style='background-color: white; border-radius: 10px; overflow: hidden;'>
                            
                            <tr>
                                <td style='background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); padding: 50px; text-align: center; color: white;'>
                                    <h1 style='margin: 0; font-size: 36px;'>Welcome to Boutique! 🎉</h1>
                                </td>
                            </tr>
                            
                            <tr>
                                <td style='padding: 40px;'>
                                    <h2 style='margin: 0 0 20px 0; color: #333;'>Hi {$name},</h2>
                                    <p style='margin: 0 0 15px 0; font-size: 16px; color: #666; line-height: 1.6;'>
                                        Thank you for joining our boutique family! We're excited to have you with us.
                                    </p>
                                    <p style='margin: 0 0 15px 0; font-size: 16px; color: #666; line-height: 1.6;'>
                                        Explore our latest collection and enjoy shopping with exclusive deals and offers.
                                    </p>
                                </td>
                            </tr>
                            
                            <tr>
                                <td style='padding: 30px; text-align: center; background: #f8f9fa;'>
                                    <a href='https://yourdomain.com' style='display: inline-block; padding: 15px 40px; background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; text-decoration: none; border-radius: 8px; font-weight: bold;'>
                                        Start Shopping
                                    </a>
                                </td>
                            </tr>
                            
                        </table>
                    </td>
                </tr>
            </table>
        </body>
        </html>
        ";
    }
}

// Usage Example:
/*
require_once 'includes/email.php';

$emailService = new EmailService();

// Send order confirmation
$emailService->sendOrderConfirmation($order, $items);

// Send status update
$emailService->sendStatusUpdate($order, 'shipped');

// Send welcome email
$emailService->sendWelcomeEmail('customer@email.com', 'John Doe');
*/

?>