<?php
require_once 'config/database.php';

$conn = getConnection();

// Get products
$category = isset($_GET['category']) ? $_GET['category'] : 'All';
$search = isset($_GET['search']) ? $_GET['search'] : '';

$sql = "SELECT * FROM products WHERE is_active = 1";
$params = [];
$types = "";

if ($category !== 'All') {
    $sql .= " AND category = ?";
    $params[] = $category;
    $types .= "s";
}

if ($search) {
    $sql .= " AND name LIKE ?";
    $params[] = "%$search%";
    $types .= "s";
}

$sql .= " ORDER BY created_at DESC";

$stmt = $conn->prepare($sql);
if (!empty($params)) {
    $stmt->bind_param($types, ...$params);
}
$stmt->execute();
$products = $stmt->get_result();

// Get categories
$categories = $conn->query("SELECT DISTINCT category FROM products WHERE is_active = 1")->fetch_all(MYSQLI_ASSOC);
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Boutique - Shop Latest Fashion</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <script src="https://cdn.tailwindcss.com"></script>
</head>
<body class="bg-gray-100 font-sans">
    <!-- Header -->
    <header class="bg-white shadow-lg sticky top-0 z-50">
        <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-4 flex flex-col sm:flex-row justify-between items-center">
            <div class="text-2xl font-bold bg-gradient-to-r from-indigo-500 to-purple-600 text-transparent bg-clip-text">Boutique</div>
            
            <form class="flex-1 max-w-md mx-4 my-2 sm:my-0 relative" method="GET">
                <i class="fas fa-search absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400"></i>
                <input type="text" name="search" placeholder="Search products..." value="<?= htmlspecialchars($search) ?>" class="w-full pl-10 pr-4 py-2 border-2 border-gray-200 rounded-full focus:border-indigo-500 focus:outline-none transition duration-300">
            </form>

            <div class="flex gap-6">
                <button class="text-gray-600 hover:text-indigo-500 transition"><i class="fas fa-heart text-lg"></i></button>
                <button class="relative text-gray-600 hover:text-indigo-500 transition" onclick="openCart()">
                    <i class="fas fa-shopping-cart text-lg"></i>
                    <span id="cartCount" class="absolute -top-2 -right-2 bg-indigo-500 text-white text-xs font-bold rounded-full w-5 h-5 flex items-center justify-center">0</span>
                </button>
                <a href="customer/login.php" class="text-gray-600 hover:text-indigo-500 transition"><i class="fas fa-user text-lg"></i></a>
            </div>
        </div>
    </header>

    <!-- Hero Banner -->
    <div class="bg-gradient-to-r from-indigo-500 to-purple-600 text-white py-16 text-center">
        <h1 class="text-4xl sm:text-5xl font-bold mb-4">New Collection Arrived</h1>
        <p class="text-lg sm:text-xl mb-6">Discover the latest trends in fashion</p>
        <p class="text-md mb-8">Free shipping on orders above ₹5,000</p>
        <button class="bg-white text-indigo-600 font-bold py-3 px-8 rounded-full hover:scale-105 transition-transform duration-300">Shop Now</button>
    </div>

    <!-- Categories -->
    <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-6 flex gap-3 overflow-x-auto">
        <a href="?category=All"><button class="px-4 py-2 bg-white rounded-full font-medium <?= $category === 'All' ? 'bg-gradient-to-r from-indigo-500 to-purple-600 text-white' : 'text-gray-600' ?> hover:shadow-lg hover:-translate-y-0.5 transition duration-300">All</button></a>
        <?php foreach ($categories as $cat): ?>
            <a href="?category=<?= urlencode($cat['category']) ?>">
                <button class="px-4 py-2 bg-white rounded-full font-medium <?= $category === $cat['category'] ? 'bg-gradient-to-r from-indigo-500 to-purple-600 text-white' : 'text-gray-600' ?> hover:shadow-lg hover:-translate-y-0.5 transition duration-300">
                    <?= htmlspecialchars($cat['category']) ?>
                </button>
            </a>
        <?php endforeach; ?>
    </div>

    <!-- Products -->
    <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
        <div class="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-6">
            <?php while ($product = $products->fetch_assoc()): ?>
                <div class="bg-white rounded-xl shadow-md overflow-hidden hover:shadow-xl hover:-translate-y-1 transition duration-300 cursor-pointer" onclick="viewProduct(<?= $product['product_id'] ?>)">
                    <div class="relative">
                        <img src="<?= htmlspecialchars($product['image_url']) ?>" alt="<?= htmlspecialchars($product['name']) ?>" class="w-full h-64 object-cover">
                        <?php if ($product['original_price']): ?>
                            <span class="absolute top-3 right-3 bg-red-500 text-white text-xs font-bold px-2 py-1 rounded-full">
                                <?= round((1 - $product['price'] / $product['original_price']) * 100) ?>% OFF
                            </span>
                        <?php endif; ?>
                    </div>
                    <div class="p-4">
                        <div class="text-lg font-bold text-gray-800 mb-1"><?= htmlspecialchars($product['name']) ?></div>
                        <div class="text-sm text-gray-500 mb-3"><?= htmlspecialchars($product['category']) ?></div>
                        <div class="flex items-center gap-2 mb-4">
                            <span class="text-xl font-bold text-indigo-600">₹<?= number_format($product['price']) ?></span>
                            <?php if ($product['original_price']): ?>
                                <span class="text-sm text-gray-400 line-through">₹<?= number_format($product['original_price']) ?></span>
                            <?php endif; ?>
                        </div>
                        <button class="w-full bg-gradient-to-r from-indigo-500 to-purple-600 text-white font-bold py-2 rounded-lg hover:shadow-lg hover:-translate-y-0.5 transition duration-300" onclick="event.stopPropagation(); addToCart(<?= $product['product_id'] ?>, '<?= htmlspecialchars($product['name']) ?>', <?= $product['price'] ?>, '<?= htmlspecialchars($product['image_url']) ?>')">
                            <i class="fas fa-shopping-cart mr-2"></i>Add to Cart
                        </button>
                    </div>
                </div>
            <?php endwhile; ?>
        </div>
    </div>

    <!-- Cart Modal -->
    <div class="modal fixed inset-0 bg-black bg-opacity-50 z-50 hidden" id="cartModal">
        <div class="bg-white w-full max-w-lg max-h-[90vh] mx-auto my-8 rounded-xl p-6 overflow-y-auto">
            <div class="flex justify-between items-center mb-6">
                <h2 class="text-2xl font-bold">Shopping Cart</h2>
                <button class="text-gray-600 hover:text-gray-800 text-2xl" onclick="closeCart()">&times;</button>
            </div>
            <div id="cartItems"></div>
            <div class="mt-4 pt-4 border-t-2 border-gray-200 text-right">
                <span class="text-xl font-bold">Total: ₹<span id="cartTotal">0</span></span>
            </div>
            <a href="customer/checkout.php"><button class="w-full bg-gradient-to-r from-indigo-500 to-purple-600 text-white font-bold py-3 rounded-lg mt-4 hover:shadow-lg transition duration-300">Proceed to Checkout</button></a>
        </div>
    </div>

    <script>
        let cart = JSON.parse(localStorage.getItem('cart')) || [];

        function updateCartDisplay() {
            document.getElementById('cartCount').textContent = cart.length;
            
            const cartItems = document.getElementById('cartItems');
            const cartTotal = document.getElementById('cartTotal');
            
            if (cart.length === 0) {
                cartItems.innerHTML = '<p class="text-center py-8 text-gray-500">Your cart is empty</p>';
                cartTotal.textContent = '0';
                return;
            }

            let html = '';
            let total = 0;

            cart.forEach((item, index) => {
                total += item.price * item.quantity;
                html += `
                    <div class="flex gap-4 py-4 border-b border-gray-200">
                        <img src="${item.image}" class="w-20 h-20 object-cover rounded-lg">
                        <div class="flex-1">
                            <strong class="text-gray-800">${item.name}</strong>
                            <div class="text-indigo-600 font-bold">₹${item.price}</div>
                            <div class="mt-2 flex items-center gap-2">
                                <button onclick="updateQuantity(${index}, -1)" class="px-2 py-1 border border-gray-300 rounded hover:bg-gray-100">-</button>
                                <span class="px-4">${item.quantity}</span>
                                <button onclick="updateQuantity(${index}, 1)" class="px-2 py-1 border border-gray-300 rounded hover:bg-gray-100">+</button>
                                <button onclick="removeItem(${index})" class="ml-4 text-red-500 hover:text-red-700"><i class="fas fa-trash"></i></button>
                            </div>
                        </div>
                    </div>
                `;
            });

            cartItems.innerHTML = html;
            cartTotal.textContent = total.toLocaleString();
        }

        function addToCart(id, name, price, image) {
            const existing = cart.find(item => item.id === id);
            if (existing) {
                existing.quantity++;
            } else {
                cart.push({ id, name, price, image, quantity: 1 });
            }
            localStorage.setItem('cart', JSON.stringify(cart));
            updateCartDisplay();
            alert('Added to cart!');
        }

        function updateQuantity(index, change) {
            cart[index].quantity += change;
            if (cart[index].quantity <= 0) {
                cart.splice(index, 1);
            }
            localStorage.setItem('cart', JSON.stringify(cart));
            updateCartDisplay();
        }

        function removeItem(index) {
            cart.splice(index, 1);
            localStorage.setItem('cart', JSON.stringify(cart));
            updateCartDisplay();
        }

        function openCart() {
            document.getElementById('cartModal').classList.add('flex');
            document.getElementById('cartModal').classList.remove('hidden');
            updateCartDisplay();
        }

        function closeCart() {
            document.getElementById('cartModal').classList.add('hidden');
            document.getElementById('cartModal').classList.remove('flex');
        }

        function viewProduct(id) {
            window.location.href = 'customer/product-details.php?id=' + id;
        }

        // Initialize
        updateCartDisplay();
    </script>
</body>
</html>